//
//	MiscAppDefaults.m -- easy access to preferences
//		Written by Steve Hayman Copyright (c) 1994 by Steve Hayman.
//				Version 1.0.  All rights reserved.
//		This notice may not be removed from this source code.
//
//	This object is included in the MiscKit by permission from the author
//	and its use is governed by the MiscKit license, found in the file
//	"LICENSE.rtf" in the MiscKit distribution.  Please refer to that file
//	for a list of all applicable permissions and restrictions.
//	

#import <appkit/appkit.h>
#import "MiscAppDefaults.h"

@implementation Application(MiscAppDefaults) 

// Three core methods first...

// Register a set of defaults under the application's name.
- (int) registerDefaults:(const NXDefaultsVector) v
{
    return NXRegisterDefaults( [NXApp appName], v );
}

// Return the requested default value as a string.
- (const char *)defaultValue:(const char *)defName
{
    return NXGetDefaultValue( [NXApp appName], defName );
}

// Set a default value.  This adds the value to the internal
// defaults cache and also to the user's .NeXT/.NeXTdefaults database.
// Returns what NXWriteDefault() returns (1 on success, 0 on failure)
- (int)setDefault:(const char *)defName to:(const char *)defValue
{
    return NXWriteDefault( [NXApp appName], defName, defValue );
}


// -----------------------------------------------------------------
// The rest of the methods are wrappers and format converters.

// Return a default value as an integer.
// Note that if the default has never been set, or has been set to
// something other than an integer, 0 will be returned.
- (int) defaultIntValue:(const char *)defName
{
    const char *v = [self defaultValue:defName];
    return( v ? atoi(v) : 0 );
}

// Return a default value as a Boolean.
// We return YES if the string value is "YES", NO if it's anything else
// (including if it has never been set.)
- (BOOL) defaultBoolValue:(const char *)defName
{
    const char *v = [self defaultValue:defName];
    return ( v ? (strcmp( v, "YES") == 0) : NO ) ;
}

// Set an integer default value.  Convert it to a string first.
- (int)setDefault:(const char *)defName toInt:(int)defValue
{
    char intBuffer[80];
    sprintf(intBuffer, "%i", defValue);
    
    return [self setDefault:defName to:intBuffer];
}

// Set a boolean default.  We write the string YES if the boolean
// is true, NO otherwise.
- (int)setDefault:(const char *)defName toBool:(BOOL)defValue
{
    return [self setDefault:defName to:(defValue ? "YES" : "NO")];
}



// For back compatability -- don't use these anymore as they will
// definitely go away in the future!
- (int)setIntDefault:(const char *)defName to:(int)defValue
{
	return [self setDefault:defName toInt:defValue];
}

- (int)setBoolDefault:(const char *)defName to:(BOOL)defValue
{
	return [self setDefault:defName toBool:defValue];
}

@end
